<?php
/**
 * moovcar functions and definitions
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package moovcar
 */

if ( ! defined( 'MVC_VERSION' ) ) {
	/*
	 * Set the theme’s version number.
	 *
	 * This is used primarily for cache busting. If you use `npm run bundle`
	 * to create your production build, the value below will be replaced in the
	 * generated zip file with a timestamp, converted to base 36.
	 */
	define( 'MVC_VERSION', 'ston7u' );
}

if ( ! defined( 'MVC_TYPOGRAPHY_CLASSES' ) ) {
	/*
	 * Set Tailwind Typography classes for the front end, block editor and
	 * classic editor using the constant below.
	 *
	 * For the front end, these classes are added by the `mvc_content_class`
	 * function. You will see that function used everywhere an `entry-content`
	 * or `page-content` class has been added to a wrapper element.
	 *
	 * For the block editor, these classes are converted to a JavaScript array
	 * and then used by the `./javascript/block-editor.js` file, which adds
	 * them to the appropriate elements in the block editor (and adds them
	 * again when they’re removed.)
	 *
	 * For the classic editor (and anything using TinyMCE, like Advanced Custom
	 * Fields), these classes are added to TinyMCE’s body class when it
	 * initializes.
	 */
	define(
		'MVC_TYPOGRAPHY_CLASSES',
		'prose prose-moovcar max-w-none prose-blockquote:pl-2 prose-blockquote:font-normal prose-li:pl-0 prose-li:my-0.25'
	);
}

if ( ! function_exists( 'mvc_setup' ) ) :
	/**
	 * Sets up theme defaults and registers support for various WordPress features.
	 *
	 * Note that this function is hooked into the after_setup_theme hook, which
	 * runs before the init hook. The init hook is too late for some features, such
	 * as indicating support for post thumbnails.
	 */
	function mvc_setup() {
		/*
		 * Make theme available for translation.
		 * Translations can be filed in the /languages/ directory.
		 * If you're building a theme based on moovcar, use a find and replace
		 * to change 'moovcar' to the name of your theme in all the template files.
		 */
		load_theme_textdomain( 'moovcar', get_template_directory() . '/languages' );

		// Add default posts and comments RSS feed links to head.
		add_theme_support( 'automatic-feed-links' );

		/*
		 * Let WordPress manage the document title.
		 * By adding theme support, we declare that this theme does not use a
		 * hard-coded <title> tag in the document head, and expect WordPress to
		 * provide it for us.
		 */
		add_theme_support( 'title-tag' );

		/*
		 * Enable support for excerpts on pages.
		 */
		add_post_type_support( 'page', 'excerpt' );

		/*
		 * Enable support for Post Thumbnails on posts and pages.
		 *
		 * @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
		 */
		add_theme_support( 'post-thumbnails' );

		// This theme uses wp_nav_menu() in two locations.
		register_nav_menus(
			array(
				'primary'             => __( 'Menu principal', 'moovcar' ),
				'footer-realisations' => __( 'Footer / Utilisations', 'moovcar' ),
				'footer-resources'    => __( 'Footer / Ressources', 'moovcar' ),
				'footer-legal'        => __( 'Footer / Documents légaux', 'moovcar' ),
			)
		);

		/*
		 * Switch default core markup for search form, comment form, and comments
		 * to output valid HTML5.
		 */
		add_theme_support(
			'html5',
			array(
				'search-form',
				'comment-form',
				'comment-list',
				'gallery',
				'caption',
				'style',
				'script',
			)
		);

		add_filter(
			'wp_editor_set_quality',
			function( $arg ) {
				return 100;
			}
		);

		add_filter(
			'jpeg_quality',
			function() {
				return 100;
			}
		);

		// Add theme support for selective refresh for widgets.
		add_theme_support( 'customize-selective-refresh-widgets' );

		// Add support for editor styles.
		add_theme_support( 'editor-styles' );

		// Enqueue editor styles.
		add_editor_style( 'style-editor.css' );
		add_editor_style( 'style-editor-extra.css' );

		// Add support for responsive embedded content.
		add_theme_support( 'responsive-embeds' );

		// Remove support for block templates.
		remove_theme_support( 'block-templates' );

		// Remove CF7 auto p feature
		add_filter( 'wpcf7_autop_or_not', '__return_false' );

		// Disable WordPress Admin Bar for all users
		add_filter( 'show_admin_bar', '__return_false' );

		// Disable image compression
		add_filter(
			'wp_editor_set_quality',
			function( $arg ) {
				return 100;
			}
		);

		// Image sizes
		add_image_size( 'realisation-large', 1440, 690, true );
		add_image_size( 'realisation-loop', 639, 445, true );
		add_image_size( 'post-large', 1074, 516, true );
		add_image_size( 'post-loop', 421, 269, true );
		add_image_size( 'page-loop', 313, 351, true );
	}
endif;
add_action( 'after_setup_theme', 'mvc_setup' );

/**
 * Register widget area.
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 */
function mvc_widgets_init() {
	register_sidebar(
		array(
			'name'          => __( 'Footer', 'moovcar' ),
			'id'            => 'sidebar-1',
			'description'   => __( 'Add widgets here to appear in your footer.', 'moovcar' ),
			'before_widget' => '<section id="%1$s" class="widget %2$s">',
			'after_widget'  => '</section>',
			'before_title'  => '<h2 class="widget-title">',
			'after_title'   => '</h2>',
		)
	);
}
add_action( 'widgets_init', 'mvc_widgets_init' );

/**
 * Enqueue scripts and styles.
 */
function mvc_scripts() {
	wp_enqueue_style( 'moovcar-style', get_stylesheet_uri(), array(), MVC_VERSION );
	wp_enqueue_script( 'jquery', 'https://cdnjs.cloudflare.com/ajax/libs/jquery/3.7.1/jquery.min.js', array(), '' );
	wp_enqueue_script( 'moovcar-script', get_template_directory_uri() . '/js/script.min.js', array( 'jquery' ), MVC_VERSION, true );

	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}
}
add_action( 'wp_enqueue_scripts', 'mvc_scripts' );

/**
 * Enqueue the block editor script.
 */
function mvc_enqueue_block_editor_script() {
	if ( is_admin() ) {
		wp_enqueue_script(
			'moovcar-editor',
			get_template_directory_uri() . '/js/block-editor.min.js',
			array(
				'wp-blocks',
				'wp-edit-post',
			),
			MVC_VERSION,
			true
		);
		wp_add_inline_script( 'moovcar-editor', "tailwindTypographyClasses = '" . esc_attr( MVC_TYPOGRAPHY_CLASSES ) . "'.split(' ');", 'before' );
	}
}
add_action( 'enqueue_block_assets', 'mvc_enqueue_block_editor_script' );

/**
 * Add the Tailwind Typography classes to TinyMCE.
 *
 * @param array $settings TinyMCE settings.
 * @return array
 */
function mvc_tinymce_add_class( $settings ) {
	$settings['body_class'] = MVC_TYPOGRAPHY_CLASSES;
	return $settings;
}
add_filter( 'tiny_mce_before_init', 'mvc_tinymce_add_class' );

/**
 * Adds an icon to menu item.
 *
 * @param string  $items Menu items.
 * @param WP_Term $args Holds the nav menu arguments.
 *
 * @return array
 */
function mvc_add_icon_to_menu_item( $items, $args ) {
	// loop
	foreach ( $items as &$item ) {
		// vars
		$icon = get_field( 'menu_icon', $item );

		// append icon
		if ( $icon ) {
			$item->title = '<div class="link-icon"><img src="' . wp_get_attachment_image_src( $icon['id'], 'full' )[0] . '" /></div><span>' . $item->title . '</span>';
		} else {
			$item->title = '<span>' . $item->title . '</span>';
		}
	}
	// return
	return $items;
}

add_filter( 'wp_nav_menu_objects', 'mvc_add_icon_to_menu_item', 10, 2 );

// Filter realisation query vars
add_action( 'pre_get_posts', 'mvc_realisation_query_vars' );
function mvc_realisation_query_vars( $query ) {
	if ( ! is_admin() && $query->is_main_query() ) {
		if ( get_query_var( 'post_type' ) === 'realisation' ) {
			$query->set(
				'posts_per_page',
				8
			);
		}
	}
	return $query;
}

/**
 * Aria Nav Menu Walker.
 */
require get_template_directory() . '/inc/aria-walker-nav-menu.php';

/**
 * A11y walker nav menu.
 */
require get_template_directory() . '/inc/a11y-walker-nav-menu.php';

/**
 * Custom template tags for this theme.
 */
require get_template_directory() . '/inc/template-tags.php';

/**
 * Functions which enhance the theme by hooking into WordPress.
 */
require get_template_directory() . '/inc/template-functions.php';
