<?php
/**
 * Functions which enhance the theme by hooking into WordPress
 *
 * @package moovcar
 */

/**
 * Add a pingback url auto-discovery header for single posts, pages, or attachments.
 */
function mvc_pingback_header() {
	if ( is_singular() && pings_open() ) {
		printf( '<link rel="pingback" href="%s">', esc_url( get_bloginfo( 'pingback_url' ) ) );
	}
}
add_action( 'wp_head', 'mvc_pingback_header' );

/**
 * Changes comment form default fields.
 *
 * @param array $defaults The default comment form arguments.
 *
 * @return array Returns the modified fields.
 */
function mvc_comment_form_defaults( $defaults ) {
	$comment_field = $defaults['comment_field'];

	// Adjust height of comment form.
	$defaults['comment_field'] = preg_replace( '/rows="\d+"/', 'rows="5"', $comment_field );

	return $defaults;
}
add_filter( 'comment_form_defaults', 'mvc_comment_form_defaults' );

/**
 * Filters the default archive titles.
 */
function mvc_get_the_archive_title() {
	if ( is_category() ) {
		$title = __( 'Category Archives: ', 'moovcar' ) . '<span>' . single_term_title( '', false ) . '</span>';
	} elseif ( is_tag() ) {
		$title = __( 'Tag Archives: ', 'moovcar' ) . '<span>' . single_term_title( '', false ) . '</span>';
	} elseif ( is_author() ) {
		$title = __( 'Author Archives: ', 'moovcar' ) . '<span>' . get_the_author_meta( 'display_name' ) . '</span>';
	} elseif ( is_year() ) {
		$title = __( 'Yearly Archives: ', 'moovcar' ) . '<span>' . get_the_date( _x( 'Y', 'yearly archives date format', 'moovcar' ) ) . '</span>';
	} elseif ( is_month() ) {
		$title = __( 'Monthly Archives: ', 'moovcar' ) . '<span>' . get_the_date( _x( 'F Y', 'monthly archives date format', 'moovcar' ) ) . '</span>';
	} elseif ( is_day() ) {
		$title = __( 'Daily Archives: ', 'moovcar' ) . '<span>' . get_the_date() . '</span>';
	} elseif ( is_post_type_archive() ) {
		$cpt   = get_post_type_object( get_queried_object()->name );
		$title = sprintf(
			/* translators: %s: Post type singular name */
			esc_html__( '%s Archives', 'moovcar' ),
			$cpt->labels->singular_name
		);
	} elseif ( is_tax() ) {
		$tax   = get_taxonomy( get_queried_object()->taxonomy );
		$title = sprintf(
			/* translators: %s: Taxonomy singular name */
			esc_html__( '%s Archives', 'moovcar' ),
			$tax->labels->singular_name
		);
	} else {
		$title = __( 'Archives:', 'moovcar' );
	}
	return $title;
}
add_filter( 'get_the_archive_title', 'mvc_get_the_archive_title' );

/**
 * Determines whether the post thumbnail can be displayed.
 */
function mvc_can_show_post_thumbnail() {
	return apply_filters( 'mvc_can_show_post_thumbnail', ! post_password_required() && ! is_attachment() && has_post_thumbnail() );
}

/**
 * Returns the size for avatars used in the theme.
 */
function mvc_get_avatar_size() {
	return 60;
}

/**
 * Create the continue reading link
 *
 * @param string $more_string The string shown within the more link.
 */
function mvc_continue_reading_link( $more_string ) {

	if ( ! is_admin() ) {
		$continue_reading = sprintf(
			/* translators: %s: Name of current post. */
			wp_kses( __( 'Continue reading %s', 'moovcar' ), array( 'span' => array( 'class' => array() ) ) ),
			the_title( '<span class="sr-only">"', '"</span>', false )
		);

		$more_string = '<a href="' . esc_url( get_permalink() ) . '">' . $continue_reading . '</a>';
	}

	return $more_string;
}

// Filter the excerpt more link.
add_filter( 'excerpt_more', 'mvc_continue_reading_link' );

// Filter the content more link.
add_filter( 'the_content_more_link', 'mvc_continue_reading_link' );

/**
 * Outputs a comment in the HTML5 format.
 *
 * This function overrides the default WordPress comment output in HTML5
 * format, adding the required class for Tailwind Typography. Based on the
 * `html5_comment()` function from WordPress core.
 *
 * @param WP_Comment $comment Comment to display.
 * @param array      $args    An array of arguments.
 * @param int        $depth   Depth of the current comment.
 */
function mvc_html5_comment( $comment, $args, $depth ) {
	$tag = ( 'div' === $args['style'] ) ? 'div' : 'li';

	$commenter          = wp_get_current_commenter();
	$show_pending_links = ! empty( $commenter['comment_author'] );

	if ( $commenter['comment_author_email'] ) {
		$moderation_note = __( 'Your comment is awaiting moderation.', 'moovcar' );
	} else {
		$moderation_note = __( 'Your comment is awaiting moderation. This is a preview; your comment will be visible after it has been approved.', 'moovcar' );
	}
	?>
<<?php echo esc_attr( $tag ); ?> id="comment-<?php comment_ID(); ?>"
	<?php comment_class( $comment->has_children ? 'parent' : '', $comment ); ?>>
	<article id="div-comment-<?php comment_ID(); ?>" class="comment-body">
		<footer class="comment-meta">
			<div class="comment-author vcard">
				<?php
				if ( 0 !== $args['avatar_size'] ) {
					echo get_avatar( $comment, $args['avatar_size'] );
				}
				?>
				<?php
					$comment_author = get_comment_author_link( $comment );

				if ( '0' === $comment->comment_approved && ! $show_pending_links ) {
					$comment_author = get_comment_author( $comment );
				}

					printf(
						/* translators: %s: Comment author link. */
						wp_kses_post( __( '%s <span class="says">says:</span>', 'moovcar' ) ),
						sprintf( '<b class="fn">%s</b>', wp_kses_post( $comment_author ) )
					);
				?>
			</div><!-- .comment-author -->

			<div class="comment-metadata">
				<?php
					printf(
						'<a href="%s"><time datetime="%s">%s</time></a>',
						esc_url( get_comment_link( $comment, $args ) ),
						esc_attr( get_comment_time( 'c' ) ),
						esc_html(
							sprintf(
							/* translators: 1: Comment date, 2: Comment time. */
								__( '%1$s at %2$s', 'moovcar' ),
								get_comment_date( '', $comment ),
								get_comment_time()
							)
						)
					);

					edit_comment_link( __( 'Edit', 'moovcar' ), ' <span class="edit-link">', '</span>' );
				?>
			</div><!-- .comment-metadata -->

			<?php if ( '0' === $comment->comment_approved ) : ?>
			<em class="comment-awaiting-moderation"><?php echo esc_html( $moderation_note ); ?></em>
			<?php endif; ?>
		</footer><!-- .comment-meta -->

		<div <?php mvc_content_class( 'comment-content' ); ?>>
			<?php comment_text(); ?>
		</div><!-- .comment-content -->

		<?php
		if ( '1' === $comment->comment_approved || $show_pending_links ) {
			comment_reply_link(
				array_merge(
					$args,
					array(
						'add_below' => 'div-comment',
						'depth'     => $depth,
						'max_depth' => $args['max_depth'],
						'before'    => '<div class="reply">',
						'after'     => '</div>',
					)
				)
			);
		}
		?>
	</article><!-- .comment-body -->
	<?php
}

/**
 * Adds option 'li_class' to 'wp_nav_menu'.
 *
 * @param string  $classes String of classes.
 * @param mixed   $item The curren item.
 * @param WP_Term $args Holds the nav menu arguments.
 *
 * @return array
 */
function mvc_nav_menu_add_li_class( $classes, $item, $args, $depth ) {
	if ( isset( $args->li_class ) ) {
		$classes[] = $args->li_class;
	}

	if ( isset( $args->{"li_class_$depth"} ) ) {
		$classes[] = $args->{"li_class_$depth"};
	}

	return $classes;
}

add_filter( 'nav_menu_css_class', 'mvc_nav_menu_add_li_class', 10, 4 );

/**
 * Adds option 'a_class' to 'wp_nav_menu'.
 *
 * @param string  $classes String of classes.
 * @param mixed   $item The curren item.
 * @param WP_Term $args Holds the nav menu arguments.
 *
 * @return array
 */
function mvc_nav_menu_add_a_class( $classes, $item, $args ) {
	if ( isset( $args->a_class ) ) {
		$classes['class'] = $args->a_class;
	}
	return $classes;
}

add_filter( 'nav_menu_link_attributes', 'mvc_nav_menu_add_a_class', 1, 3 );

/**
 * Returns an array of allowed HTML tags.
 */
function mvc_allowed_html() {

	$allowed_tags = array(
		'a'          => array(
			'class' => array(),
			'href'  => array(),
			'rel'   => array(),
			'title' => array(),
			'style' => array(),
		),
		'abbr'       => array(
			'title' => array(),
		),
		'b'          => array(),
		'blockquote' => array(
			'cite'  => array(),
			'style' => array(),
		),
		'br'         => array(
			'class' => array(),
		),
		'cite'       => array(
			'title' => array(),
		),
		'code'       => array(),
		'del'        => array(
			'datetime' => array(),
			'title'    => array(),
		),
		'dd'         => array(),
		'div'        => array(
			'class' => array(),
			'title' => array(),
			'style' => array(),
		),
		'dl'         => array(),
		'dt'         => array(),
		'em'         => array(),
		'h1'         => array(
			'style' => array(),
		),
		'h2'         => array(
			'style' => array(),
		),
		'h3'         => array(
			'style' => array(),
		),
		'h4'         => array(
			'style' => array(),
		),
		'h5'         => array(
			'style' => array(),
		),
		'h6'         => array(
			'style' => array(),
		),
		'i'          => array(),
		'iframe'     => array(
			'title'           => array(),
			'width'           => array(),
			'height'          => array(),
			'src'             => array(),
			'allow'           => array(),
			'referrerpolicy'  => array(),
			'allowfullscreen' => array(),
		),
		'img'        => array(
			'alt'    => array(),
			'class'  => array(),
			'height' => array(),
			'src'    => array(),
			'width'  => array(),
		),
		'li'         => array(
			'class' => array(),
			'style' => array(),
		),
		'ol'         => array(
			'class' => array(),
			'style' => array(),
		),
		'p'          => array(
			'class' => array(),
			'style' => array(),
		),
		'q'          => array(
			'cite'  => array(),
			'title' => array(),
		),
		'span'       => array(
			'class' => array(),
			'title' => array(),
			'style' => array(),
			'style' => array(),
		),
		'strike'     => array(),
		'strong'     => array(),
		'sup'        => array(),
		'ul'         => array(
			'class' => array(),
			'style' => array(),
		),
	);

	return $allowed_tags;
}

/**
 * Processes a block list output from the parse_blocks() function
 * and returns a flat list of block arrays matching the specified
 * type.
 *
 * @param array  $blocks Output from parse_blocks().
 * @param string $type The block type handle to search for.
 * @return array An array of all block arrays with the specified type.
 **/
function mvc_find_blocks( $blocks, $type ) {
	$matches = array();

	foreach ( $blocks as $block ) {
		if ( $block['blockName'] === $type ) {
			$matches[] = $block;
		}

		if ( count( $block['innerBlocks'] ) ) {
			$matches = array_merge(
				$matches,
				mvc_find_blocks( $block['innerBlocks'], $type )
			);
		}
	}

	return $matches;
}

function mvc_excerpt_length( $length ) {
	return 20;
}
add_filter( 'excerpt_length', 'mvc_excerpt_length' );

function mvc_excerpt_more( $length ) {
	return '...';
}
add_filter( 'excerpt_more', 'mvc_excerpt_more' );
